--------------------------------------------------------------------
-- |
-- Module      : Flickr.Photos.Geo
-- Description : flickr.photos.geo - setting/getting photo geo location.
-- Copyright   : (c) Sigbjorn Finne, 2008
-- License     : BSD3
--
-- Maintainer  : Sigbjorn Finne <sof@forkIO.com>
-- Stability   : provisional
-- Portability : portable
--
-- flickr.photos.geo API, setting/getting photo geo location.
--------------------------------------------------------------------
module Flickr.Photos.Geo where

import Flickr.Monad
import Flickr.Utils
import Flickr.Types
import Flickr.Types.Import

-- | Get the geo data (latitude and longitude and the accuracy level) for a photo.
getLocation :: PhotoID -> FM GeoLocation
getLocation pid = 
  flickTranslate toGeoLocation $
    flickrCall "flickr.photos.geo.getLocation"
               [ ("photo_id", pid) ]

-- | Get permissions for who may view geo data for a photo.
getPerms :: PhotoID -> FM Permissions
getPerms pid = 
  flickTranslate toPermissions $
    flickrCall "flickr.photos.geo.getPerms"
               [ ("photo_id", pid) ]

-- | Removes the geo data associated with a photo.
removeLocation :: PhotoID -> FM ()
removeLocation pid = withWritePerm $ postMethod $
    flickCall_ "flickr.photos.geo.removeLocation"
               [ ("photo_id", pid) ]

-- | Sets the geo data (latitude and longitude and, optionally,
-- the accuracy level) for a photo. Before users may assign
-- location data to a photo they must define who, by default,
-- may view that information. Users can edit this preference
-- at http://www.flickr.com/account/geo/privacy/. If a user
-- has not set this preference, the API method will return 
-- an error.
setLocation :: PhotoID -> GeoLocation -> FM ()
setLocation pid (la,lo,ac) = withWritePerm $ postMethod $ 
    flickCall_ "flickr.photos.geo.setLocation"
                 [ ("photo_id", pid)
	         , ("lat", la)
	         , ("lon", lo)
		 , ("accuracy", show ac)
	         ]

-- | Set the permission for who may view the geo data associated with a photo.
setPerms :: PhotoID -> Permissions -> FM ()
setPerms pid p = withWritePerm $ postMethod $ 
    flickCall_ "flickr.photos.geo.setPerms"
               [ ("photo_id", pid)
	       , ("is_public",  showBool $ permIsPublic p)
	       , ("is_friend",  showBool $ permIsFriend p)
	       , ("is_family",  showBool $ permIsFamily p)
	       , ("is_contact", showBool (permIsFamily p || permIsFriend p))
	       ]



